﻿using System;
using System.Collections.Generic;
using System.Data;
using System.Linq;
using System.Web;
using System.Web.UI;

using Telerik.Web.UI;

using VeteransAffairs.Registries.BusinessManager;

public partial class PatientImaging : ImagingBasePage
{
    private readonly UserAccountManager _currentUser;
    private readonly PatientImagingManager _patientImagingManager;

    private static IReadOnlyList<string> _navigationSources = new List<string>
    {
        "/Patients.aspx",
        "/Patient.aspx",
        "/PatientWorkflows.aspx",
        "/PatientRelatedDiagnoses.aspx",
        "/PatientRelatedLabTests.aspx",
    };

    private static readonly IEnumerable<string> _referenceImageFragmentSizes;

    static PatientImaging()
    {
        _referenceImageFragmentSizes = PatientImagingManager.LoadReferenceImageFragmentSizes();
    }

    public PatientImaging()
    {
        _currentUser = (UserAccountManager)HttpContext.Current.User;
        _patientImagingManager = new PatientImagingManager();

        _patientImagingManager.BOSaveSuccess += new BaseBO.BOSaveSuccessEvent(manager_BOSaveSuccess);
    }

    protected void Page_Load(object sender, EventArgs e)
    {
        if (ViewState[OriginallyNavigatedFrom] == null)
            ViewState[OriginallyNavigatedFrom] = Request.UrlReferrer;

        if (!IsPostBack)
        {
            var patientSummary = GetOrStorePatientSummary();
            if (patientSummary != null)
            {
                PopulatControls();

                SetBreadCrumbLabel(patientSummary);

                LoadData(patientSummary.Icn);
            }
        }

        RadAjaxManager1.AjaxRequest += new RadAjaxControl.AjaxRequestDelegate(RadAjaxManager1_AjaxRequest);
        RadGridImages.NeedDataSource += new GridNeedDataSourceEventHandler(RadGrid1_NeedDataSource);

        ScriptManager scriptManager = ScriptManager.GetCurrent(Page);
        scriptManager.RegisterPostBackControl(OKButton);

        RadComboBoxInCprs.Focus();
    }

    protected void RadAjaxManager1_AjaxRequest(object sender, AjaxRequestEventArgs e)
    {
        if (e.Argument == "RebindLatePanel")
            RadGridImages.Rebind();
    }

    protected void RadGrid1_NeedDataSource(object source, GridNeedDataSourceEventArgs e)
    {
        if (Page.IsPostBack)
            RadGridImages.DataSource = PatientImageData;
    }

    protected void ScriptManager1_AsyncPostBackError(object sender, AsyncPostBackErrorEventArgs e)
    {
        ScriptManager1.AsyncPostBackErrorMessage = "VistA Imaging Services are not currently available.  Please try again later.";
    }

    public DataTable PatientImageData
    {
        get
        {
            var patientSummary = GetOrStorePatientSummary();
            if (patientSummary != null)
            {
                if (_patientImagingManager.ShouldRefreshImageData(patientSummary.Id))
                    _patientImagingManager.RefreshImageData(patientSummary.Icn, patientSummary.Id);

                return _patientImagingManager.GetPatientImagesDataTable(patientSummary.Icn);
            }

            return null;
        }
    }

    private void SetBreadCrumbLabel(PatientSummary patientSummary)
    {
        BreadCrumbLabel.Text = $"{BreadCrumbLabel.Text}{patientSummary.FullName}";
        LabelPatientName.Text = patientSummary.FullName;
        LabelPatientSnum.Text = patientSummary.Snum;
    }

    private void PopulatControls()
    {
        PopulateBooleanRadComboBoxes();
        PopulateSpecializedRadComboBoxes();
    }

    private void PopulateBooleanRadComboBoxes()
    {
        foreach (var combo in GetBooleanRadComboBoxes())
        {
            combo.Items.Add(new RadComboBoxItem("(Select a Value)"));
            combo.Items.Add(new RadComboBoxItem("Yes"));
            combo.Items.Add(new RadComboBoxItem("No"));
        }
    }

    private void PopulateSpecializedRadComboBoxes()
    {
        RadComboBoxFragmentSize.Items.Add(new RadComboBoxItem("(Select a Value)"));
        foreach (var item in _referenceImageFragmentSizes)
            RadComboBoxFragmentSize.Items.Add(new RadComboBoxItem(item));
    }

    private IEnumerable<RadComboBox> GetBooleanRadComboBoxes()
    {
        return new List<RadComboBox>
        {
            RadComboBoxInCprs,
            RadComboBoxFragmentVerified,
            RadComboBoxBoneFracture,
            RadComboBoxJointSpace
        };
    }

    private void ResetRadComboBoxes()
    {
        foreach (var combo in GetBooleanRadComboBoxes())
            combo.SelectedIndex = 0;

        RadComboBoxFragmentSize.SelectedIndex = 0;
    }

    protected void SaveButton_Click(object sender, EventArgs e)
    {
        if (Page.IsValid)
        {
            var patientSummary = GetOrStorePatientSummary();

            var param = GetUpdateParameters(patientSummary.Icn);

            _patientImagingManager.InsertUpdateImageAnalysis(param);

            UpdateNavigation();

            LoadData(patientSummary.Icn);
        }
    }

    protected void DeleteButton_Click(object sender, EventArgs e)
    {
        var patientSummary = GetOrStorePatientSummary();

        _patientImagingManager.DeleteImageAnalysis(patientSummary.Icn);

        UpdateNavigation();

        LoadData(patientSummary.Icn);
    }

    protected void BackButton_Click(object sender, EventArgs e)
    {
        Uri from = ViewState[OriginallyNavigatedFrom] as Uri;
        if (from != null && _navigationSources.Contains(from.LocalPath))
            Response.Redirect(from.PathAndQuery);

        Response.Redirect("Patients.aspx");
    }

    private void UpdateNavigation()
    {
        object countField = ViewState["PageSaveCount"];
        var count = countField != null ? Convert.ToInt32(countField) + 1 : 1;
        ViewState["PageSaveCount"] = count;
        PageSaveCount.Value = count.ToString();
    }

    private object[] GetUpdateParameters(string patientIcn)
    {
        return new object[]
        {
            patientIcn,
            GetDateTimeValueFromPicker(RadDatePickerDateImagingReviewedByStaff),
            GetRadComboBoxBooleanValue(RadComboBoxInCprs),
            GetRadComboBoxBooleanValue(RadComboBoxFragmentVerified),
            GetRadComboBoxBooleanValue(RadComboBoxBoneFracture),
            GetRadComboBoxBooleanValue(RadComboBoxJointSpace),
            GetValueFromRadTextBox(RadTextBoxJoint),
            GetValueFromRadTextBox(RadTextBoxOtherObservations),
            GetRadComboBoxSelectedIndex(RadComboBoxFragmentSize),
            _currentUser.UserName
        };
    }

    private bool? GetRadComboBoxBooleanValue(RadComboBox combo)
    {
        return combo.SelectedIndex > 0 ? combo.SelectedIndex == 1 : (bool?)null;
    }

    private int? GetRadComboBoxSelectedIndex(RadComboBox combo)
    {
        int? result = null;

        if (combo.SelectedIndex > 0)
            result = combo.SelectedIndex;

        return result;
    }

    private DateTime? GetDateTimeValueFromPicker(RadDatePicker datePicker)
    {
        DateTime? result = null;

        if (datePicker.SelectedDate.HasValue)
            result = datePicker.SelectedDate.Value;

        return result;
    }

    private string GetValueFromRadTextBox(RadTextBox textBox)
    {
        return string.IsNullOrEmpty(textBox.Text) ? null : textBox.Text;
    }

    private void LoadData(string patientIcn)
    {
        PatientImageAnalysis patientImageAnalysis = _patientImagingManager.GetPatientImageAnalysis(patientIcn);
        if (patientImageAnalysis != null)
        {
            if (patientImageAnalysis.ImagesInCprsFlag.HasValue)
                RadComboBoxInCprs.SelectedIndex = patientImageAnalysis.ImagesInCprsFlag.Value ? 1 : 2;
            else
                RadComboBoxInCprs.SelectedIndex = 0;

            if (patientImageAnalysis.FragmentVerifiedFlag.HasValue)
            {
                RadComboBoxFragmentVerified.SelectedIndex = patientImageAnalysis.FragmentVerifiedFlag.Value ? 1 : 2;
                LabelFragmentVerifiedHeader.Text = RadComboBoxFragmentVerified.Items[RadComboBoxFragmentVerified.SelectedIndex].Text;
            }
            else
            {
                RadComboBoxFragmentVerified.SelectedIndex = 0;
                LabelFragmentVerifiedHeader.Text = string.Empty;
            }

            if (patientImageAnalysis.BoneFractureDocumentedFlag.HasValue)
                RadComboBoxBoneFracture.SelectedIndex = patientImageAnalysis.BoneFractureDocumentedFlag.Value ? 1 : 2;
            else
                RadComboBoxBoneFracture.SelectedIndex = 0;

            if (patientImageAnalysis.FragmentsInJointSpaceFlag.HasValue)
                RadComboBoxJointSpace.SelectedIndex = patientImageAnalysis.FragmentsInJointSpaceFlag.Value ? 1 : 2;
            else
                RadComboBoxJointSpace.SelectedIndex = 0;

            if (patientImageAnalysis.StandardImageFragmentSizeId.HasValue)
                RadComboBoxFragmentSize.Items[patientImageAnalysis.StandardImageFragmentSizeId.Value].Selected = true;
            else
                RadComboBoxFragmentSize.SelectedIndex = 0;

            if (!string.IsNullOrEmpty(patientImageAnalysis.JointSpace))
                RadTextBoxJoint.Text = patientImageAnalysis.JointSpace;
            else
                RadTextBoxJoint.Text = string.Empty;

            if (!string.IsNullOrEmpty(patientImageAnalysis.OtherObservations))
                RadTextBoxOtherObservations.Text = patientImageAnalysis.OtherObservations;
            else
                RadTextBoxOtherObservations.Text = string.Empty;

            if (patientImageAnalysis.DateReviewed.HasValue)
                RadDatePickerDateImagingReviewedByStaff.DbSelectedDate = patientImageAnalysis.DateReviewed;
            else
                RadDatePickerDateImagingReviewedByStaff.DbSelectedDate = null;

            if (RadDatePickerDateImagingReviewedByStaff.SelectedDate.HasValue)
                LabelImagingReviewedHeader.Text = RadDatePickerDateImagingReviewedByStaff.SelectedDate.Value.ToString(DateFormat);
            else
                LabelImagingReviewedHeader.Text = string.Empty;

            CheckForHistory(patientIcn);
        }
        else
        {
            ResetRadComboBoxes();

            RadTextBoxJoint.Text = string.Empty;
            RadTextBoxOtherObservations.Text = string.Empty;

            RadDatePickerDateImagingReviewedByStaff.DbSelectedDate = null;

            LabelImagingReviewedHeader.Text = string.Empty;
            LabelFragmentVerifiedHeader.Text = string.Empty;
        }
    }

    private void CheckForHistory(string patientId)
    {
        var history = _patientImagingManager.ImageAnalysisHistoryExists(patientId);
        if (history)
            HiddenFieldHasHistory.Value = "1";
        else
            HiddenFieldHasHistory.Value = null;
    }

    protected string GetBodyPartsImaged(object patientImageId)
    {
        var id = Convert.ToInt32(patientImageId);

        return _patientImagingManager.GetBodyPartsImaged(id);
    }

}
